/*
 * Copyright (C) 2014-2017 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "ShapeValue.h"

#include "AnimationUtilities.h"
#include "CachedImage.h"
#include <wtf/PointerComparison.h>

namespace WebCore {

bool ShapeValue::isImageValid() const
{
    if (!m_image)
        return false;
    if (m_image->hasCachedImage()) {
        auto* cachedImage = m_image->cachedImage();
        return cachedImage && cachedImage->hasImage();
    }
    return m_image->isGeneratedImage();
}

bool ShapeValue::operator==(const ShapeValue& other) const
{
    return m_type == other.m_type
        && m_cssBox == other.m_cssBox
        && arePointingToEqualData(m_shape, other.m_shape)
        && arePointingToEqualData(m_image, other.m_image);
}

bool ShapeValue::canBlend(const ShapeValue& to) const
{
    if (m_type != ShapeValue::Type::Shape || to.type() != ShapeValue::Type::Shape)
        return false;

    if (m_cssBox != to.cssBox())
        return false;

    if (RefPtr toShape = to.shape())
        return m_shape && m_shape->canBlend(*toShape);

    return false;
}

Ref<ShapeValue> ShapeValue::blend(const ShapeValue& to, const BlendingContext& context) const
{
    ASSERT(m_shape && to.shape());
    return ShapeValue::create(to.shape()->blend(*m_shape, context), m_cssBox);
}

CSSBoxType ShapeValue::effectiveCSSBox() const
{
    if (m_cssBox == CSSBoxType::BoxMissing)
        return m_type == ShapeValue::Type::Image ? CSSBoxType::ContentBox : CSSBoxType::MarginBox;
    return m_cssBox;
}

} // namespace WebCore
